import type { Metadata } from "next";
import { setRequestLocale } from "next-intl/server";
import { Navbar } from "@/components/navbar";
import { Footer } from "@/modules/landing/components";
import { getServices } from "@/core/lib/api";
import { buildMetadataFromSeo } from "@/core/lib/seo";

type Props = {
  params: Promise<{ locale: string; slug: string }>;
};

export async function generateMetadata({ params }: Props): Promise<Metadata> {
  const { locale, slug } = await params;
  const services = await getServices({ locale });
  const service = services.find((s) => s.slug === slug);

  if (!service) {
    return buildMetadataFromSeo({
      locale: (locale === "ar" ? "ar" : "en") as "en" | "ar",
      seo: {
        metaTitle: "Service Not Found | YS Lootah Tech",
        metaDescription:
          "The requested service could not be found. Discover YS Lootah Tech services in software development, cloud, AI, cybersecurity, and digital transformation.",
      },
    });
  }

  return buildMetadataFromSeo({
    seo: service.seo ?? undefined,
    openGraph: service.openGraph ?? undefined,
    locale: (locale === "ar" ? "ar" : "en") as "en" | "ar",
    override: {
      title: service.name,
    },
  });
}

export default async function ServicePage({ params }: Props) {
  const { locale, slug } = await params;
  setRequestLocale(locale);

  const services = await getServices({ locale });
  const service = services.find((s) => s.slug === slug);

  return (
    <div className="min-h-screen bg-slate-950">
      <Navbar />
      <main className="container py-16">
        {service ? (
          <div>
            <h1 className="text-3xl font-bold text-white mb-4">{service.name}</h1>
            {service.short_description && (
              <p className="text-slate-300 max-w-3xl">{service.short_description}</p>
            )}
          </div>
        ) : (
          <p className="text-slate-300">Service not found.</p>
        )}
      </main>
      <Footer />
    </div>
  );
}
